# Company Filtering Feature

## Overview
The jobs page now supports filtering by specific company using the `company` URL parameter. When this parameter is provided, the page will display only jobs from that particular company.

## Usage

### URL Format
```
http://localhost/recruit/jobs.php?company=8
```

Where `8` is the company ID.

### Features

#### 1. **Company-Specific Job Display**
- Shows only jobs from the specified company
- Works with all existing filters (category, location, salary, etc.)
- Maintains pagination and sorting functionality

#### 2. **Dynamic Page Updates**
- **Page Title**: Changes to "Jobs at [Company Name] - Recruitment Portal"
- **Header**: Updates to show "Jobs at [Company Name]"
- **Results Count**: Displays "X jobs found at [Company Name]"
- **Active Filters**: Shows company name as an active filter

#### 3. **Security & Validation**
- Validates company ID exists and is approved
- Redirects to main jobs page for invalid company IDs
- Only shows jobs from approved companies
- Respects job visibility settings (internal/external/both)

#### 4. **Filter Integration**
- Company filter works with all existing filters
- Company parameter is preserved when using other filters
- Can be combined with keyword searches, location filters, etc.

## Implementation Details

### Code Changes

#### 1. Parameter Handling (`jobs.php:18`)
```php
$company_filter = isset($_GET['company']) ? (int)$_GET['company'] : 0;
```

#### 2. Company Lookup (`jobs.php:20-31`)
```php
// Get company information if filtering by company
$filtered_company = null;
if ($company_filter) {
    $stmt = $db->prepare("SELECT id, name, logo FROM companies WHERE id = ? AND status = 'approved'");
    $stmt->execute([$company_filter]);
    $filtered_company = $stmt->fetch();
    
    // If company doesn't exist or is not approved, redirect to all jobs
    if (!$filtered_company) {
        redirect('jobs.php');
    }
}
```

#### 3. Query Filter (`jobs.php:83-87`)
```php
// Company filter - show only jobs from specific company if requested
if ($company_filter && $filtered_company) {
    $where_conditions[] = "j.company_id = ?";
    $params[] = $company_filter;
}
```

#### 4. UI Updates
- **Page Title** (`jobs.php:290`): Dynamic title based on company
- **Header** (`jobs.php:625-641`): Company-specific header
- **Results Count** (`jobs.php:1069-1070`): Shows company name in results
- **Active Filters** (`jobs.php:1090`): Company name as active filter
- **Form Fields** (`jobs.php:688`): Preserves company parameter in forms

## Testing

The implementation has been tested with:
- ✅ Valid company IDs (shows correct jobs)
- ✅ Invalid company IDs (redirects properly)
- ✅ Company lookup and validation
- ✅ Filter integration
- ✅ UI updates and display

### Test Results
- **Company ID 2 (PetroSA)**: 9 jobs found
- **Company ID 8 (SANPC)**: 2 jobs found
- **Invalid ID 99999**: Redirects to main page

## Examples

### Basic Company Filter
```
http://localhost/recruit/jobs.php?company=8
```
Shows all jobs from company ID 8.

### Company Filter with Search
```
http://localhost/recruit/jobs.php?company=8&keyword=developer
```
Shows jobs from company ID 8 containing the word "developer".

### Company Filter with Multiple Filters
```
http://localhost/recruit/jobs.php?company=8&category=1&employment_type=full-time
```
Shows full-time jobs in category 1 from company ID 8.

## Error Handling

- **Invalid Company ID**: Redirects to `jobs.php` 
- **Non-existent Company**: Redirects to `jobs.php`
- **Unapproved Company**: Redirects to `jobs.php`
- **No Jobs Found**: Shows "No jobs found" message with company context

## Benefits

1. **Direct Company Access**: Companies can share direct links to their job listings
2. **Better User Experience**: Focused view of company-specific opportunities  
3. **Filter Compatibility**: Works seamlessly with existing search and filter system
4. **SEO Friendly**: Clean URLs with meaningful company-specific titles
5. **Secure**: Proper validation and error handling

This feature enhances the job browsing experience by allowing users to view jobs from specific companies while maintaining all existing functionality.