<?php
// Scoring system helper functions

// Get predefined response scores based on keywords
function getResponseScore($response, $question_type = "text") {
    $response = strtolower(trim($response));
    
    // Enhanced keyword recognition system
    $scoring_map = [
        // 100% Points (Full Weight)
        "excellent" => 100, "yes" => 100, "expert" => 100, "advanced" => 100,
        "very good" => 100, "6+" => 100, "9+" => 100, "10+" => 100,
        "10+ years" => 100, "more than 10" => 100, "over 10" => 100,
        
        // 70% Points
        "good" => 70, "satisfactory" => 70, "intermediate" => 70,
        "3-5" => 70, "4-6" => 70, "3-5 years" => 70, "4-6 years" => 70,
        
        // 50% Points
        "fair" => 50, "average" => 50, "basic" => 50, "basic knowledge" => 50,
        "1-3" => 50, "2-4" => 50, "1-3 years" => 50, "2-4 years" => 50,
        
        // 0% Points
        "no" => 0, "none" => 0, "never" => 0, "not applicable" => 0,
        "no experience" => 0, "0" => 0, "0 years" => 0
    ];
    
    // Check for exact matches first
    if (isset($scoring_map[$response])) {
        return $scoring_map[$response];
    }
    
    // Check for partial matches
    foreach ($scoring_map as $keyword => $score) {
        if (strpos($response, $keyword) !== false) {
            return $score;
        }
    }
    
    // Check for numeric years experience
    if (preg_match("/(\d+)(?:\s*(?:years?|yrs?))?/", $response, $matches)) {
        $years = intval($matches[1]);
        if ($years >= 10) return 100;
        if ($years >= 6) return 100;
        if ($years >= 3) return 70;
        if ($years >= 1) return 50;
        return 0;
    }
    
    // If response is not empty but unrecognized, give partial credit
    if (!empty($response)) {
        return 25; // Partial credit for any response
    }
    
    return 0; // Empty response
}

// Calculate weighted score using percentage weights
function calculateWeightedScore($response_score, $weight_percentage) {
    // weight_percentage is already a percentage (0-100)
    // response_score is 0-100 (percentage of question answered correctly)
    // Final contribution = (response_score / 100) * weight_percentage
    return ($response_score / 100) * $weight_percentage;
}

// Calculate total application score using weighted percentages
function calculateApplicationScore($application_id, $db) {
    $stmt = $db->prepare("
        SELECT ars.*, jsq.weight 
        FROM application_scores ars
        JOIN job_survey_questions jsq ON ars.question_id = jsq.id
        WHERE ars.application_id = ?
    ");
    $stmt->execute([$application_id]);
    $scores = $stmt->fetchAll();
    
    if (empty($scores)) {
        return [
            "weighted_score" => 0,
            "total_questions" => 0,
            "percentage" => 0
        ];
    }
    
    $total_weighted_score = 0;
    $total_weight = 0;
    $question_count = count($scores);
    
    foreach ($scores as $score) {
        $weight = (float)$score["weight"]; // This is now a percentage
        $response_score = (float)$score["score_earned"]; // 0-100
        
        // Calculate weighted contribution
        $weighted_contribution = calculateWeightedScore($response_score, $weight);
        $total_weighted_score += $weighted_contribution;
        $total_weight += $weight;
    }
    
    // The total_weighted_score should equal the final percentage (since weights total 100%)
    return [
        "weighted_score" => round($total_weighted_score, 2),
        "total_questions" => $question_count,
        "percentage" => round($total_weighted_score, 2)
    ];
}

// Score categories for display
function getScoreCategory($percentage) {
    if ($percentage >= 80) return ["category" => "Excellent", "class" => "success", "icon" => "star"];
    if ($percentage >= 60) return ["category" => "Good", "class" => "primary", "icon" => "thumbs-up"];
    if ($percentage >= 40) return ["category" => "Average", "class" => "warning", "icon" => "minus-circle"];
    return ["category" => "Poor", "class" => "danger", "icon" => "times-circle"];
}
?>