<?php
require_once 'config/config.php';

$database = new Database();
$db = $database->getConnection();

echo "<h2>🧪 Test Deadline Functionality</h2>\n";

// 1. Check current deadline status
echo "<h3>1. Current Job Deadline Status</h3>\n";

$stmt = $db->query("
    SELECT 
        id, title, deadline, status,
        CASE 
            WHEN deadline IS NULL THEN 'No Deadline'
            WHEN deadline < CURDATE() THEN 'Expired'
            WHEN deadline = CURDATE() THEN 'Expires Today'
            WHEN deadline <= DATE_ADD(CURDATE(), INTERVAL 7 DAY) THEN 'Expires Soon'
            ELSE 'Active'
        END as deadline_status,
        DATEDIFF(deadline, CURDATE()) as days_remaining
    FROM jobs 
    ORDER BY deadline ASC
");
$jobs = $stmt->fetchAll();

echo "<table border='1' cellpadding='5'>\n";
echo "<tr><th>ID</th><th>Title</th><th>Deadline</th><th>Status</th><th>Days Left</th><th>Should Show?</th></tr>\n";

foreach ($jobs as $job) {
    $should_show = ($job['deadline_status'] !== 'Expired');
    $row_color = '';
    
    switch ($job['deadline_status']) {
        case 'Expired':
            $row_color = 'style="background-color: #ffebee;"';
            break;
        case 'Expires Today':
            $row_color = 'style="background-color: #fff3cd;"';
            break;
        case 'Expires Soon':
            $row_color = 'style="background-color: #fff3cd;"';
            break;
        case 'Active':
            $row_color = 'style="background-color: #e8f5e8;"';
            break;
        default:
            $row_color = 'style="background-color: #f0f0f0;"';
            break;
    }
    
    echo "<tr $row_color>";
    echo "<td>{$job['id']}</td>";
    echo "<td>" . substr($job['title'], 0, 40) . "...</td>";
    echo "<td>" . ($job['deadline'] ?: 'N/A') . "</td>";
    echo "<td>{$job['deadline_status']}</td>";
    echo "<td>" . ($job['days_remaining'] !== null ? $job['days_remaining'] : 'N/A') . "</td>";
    echo "<td>" . ($should_show ? '✅ YES' : '❌ NO') . "</td>";
    echo "</tr>\n";
}
echo "</table>\n";

// 2. Test job listing filtering (jobs.php simulation)
echo "<h3>2. Job Listing Filtering Test</h3>\n";

$stmt = $db->query("
    SELECT j.id, j.title, j.deadline, j.status,
           CASE 
               WHEN j.deadline IS NULL OR j.deadline >= CURDATE() THEN 'Would Show'
               ELSE 'Would Hide'
           END as listing_visibility
    FROM jobs j 
    LEFT JOIN companies c ON j.company_id = c.id 
    WHERE j.status = 'active' AND c.status = 'approved'
    ORDER BY j.deadline ASC
");
$listing_test = $stmt->fetchAll();

echo "<h4>Jobs.php Filtering Results:</h4>\n";
echo "<table border='1' cellpadding='5'>\n";
echo "<tr><th>ID</th><th>Title</th><th>Deadline</th><th>Visibility</th></tr>\n";

foreach ($listing_test as $job) {
    $color = ($job['listing_visibility'] === 'Would Show') ? 'style="background-color: #e8f5e8;"' : 'style="background-color: #ffebee;"';
    echo "<tr $color>";
    echo "<td>{$job['id']}</td>";
    echo "<td>" . substr($job['title'], 0, 40) . "...</td>";
    echo "<td>" . ($job['deadline'] ?: 'N/A') . "</td>";
    echo "<td>{$job['listing_visibility']}</td>";
    echo "</tr>\n";
}
echo "</table>\n";

// 3. Test filtered query (what users actually see)
echo "<h3>3. What Users Actually See</h3>\n";

$stmt = $db->query("
    SELECT j.id, j.title, j.deadline, j.status
    FROM jobs j 
    LEFT JOIN companies c ON j.company_id = c.id 
    WHERE j.status = 'active' AND c.status = 'approved' AND (j.deadline IS NULL OR j.deadline >= CURDATE())
    ORDER BY j.created_at DESC
");
$visible_jobs = $stmt->fetchAll();

echo "<h4>Jobs visible to users (" . count($visible_jobs) . " jobs):</h4>\n";
if ($visible_jobs) {
    echo "<table border='1' cellpadding='5'>\n";
    echo "<tr><th>ID</th><th>Title</th><th>Deadline</th></tr>\n";
    foreach ($visible_jobs as $job) {
        echo "<tr style='background-color: #e8f5e8;'>";
        echo "<td>{$job['id']}</td>";
        echo "<td>" . substr($job['title'], 0, 50) . "...</td>";
        echo "<td>" . ($job['deadline'] ?: 'No deadline') . "</td>";
        echo "</tr>\n";
    }
    echo "</table>\n";
} else {
    echo "<p>No jobs are currently visible to users.</p>\n";
}

// 4. Test individual job access
echo "<h3>4. Individual Job Access Test</h3>\n";

// Test a few specific jobs
$test_jobs = [7, 14, 15, 16]; // Mix of expired and active jobs
foreach ($test_jobs as $job_id) {
    $stmt = $db->prepare("SELECT id, title, deadline, status FROM jobs WHERE id = ?");
    $stmt->execute([$job_id]);
    $job = $stmt->fetch();
    
    if ($job) {
        $is_expired = $job['deadline'] && date('Y-m-d') > $job['deadline'];
        $can_apply = !$is_expired && $job['status'] === 'active';
        
        echo "<div style='background: #f8f9fa; padding: 10px; margin: 5px 0; border-radius: 5px;'>\n";
        echo "<h4>Job ID {$job['id']}: " . substr($job['title'], 0, 50) . "...</h4>\n";
        echo "<p><strong>Deadline:</strong> " . ($job['deadline'] ?: 'No deadline') . "</p>\n";
        echo "<p><strong>Status:</strong> {$job['status']}</p>\n";
        echo "<p><strong>Is Expired:</strong> " . ($is_expired ? "❌ YES" : "✅ NO") . "</p>\n";
        echo "<p><strong>Can Apply:</strong> " . ($can_apply ? "✅ YES" : "❌ NO") . "</p>\n";
        echo "</div>\n";
    }
}

// 5. Summary statistics
echo "<h3>5. Summary Statistics</h3>\n";

$stmt = $db->query("
    SELECT 
        COUNT(*) as total_jobs,
        SUM(CASE WHEN deadline IS NULL OR deadline >= CURDATE() THEN 1 ELSE 0 END) as visible_jobs,
        SUM(CASE WHEN deadline IS NOT NULL AND deadline < CURDATE() THEN 1 ELSE 0 END) as expired_jobs,
        SUM(CASE WHEN deadline IS NULL THEN 1 ELSE 0 END) as no_deadline_jobs
    FROM jobs 
    WHERE status = 'active'
");
$stats = $stmt->fetch();

echo "<div style='background: #e3f2fd; color: #1565c0; padding: 15px; border-radius: 5px; margin: 10px 0;'>\n";
echo "<h4>📊 Implementation Results:</h4>\n";
echo "<ul>\n";
echo "<li><strong>Total Active Jobs:</strong> {$stats['total_jobs']}</li>\n";
echo "<li><strong>Visible to Users:</strong> {$stats['visible_jobs']} ✅</li>\n";
echo "<li><strong>Hidden (Expired):</strong> {$stats['expired_jobs']} ❌</li>\n";
echo "<li><strong>No Deadline Set:</strong> {$stats['no_deadline_jobs']} 📅</li>\n";
echo "</ul>\n";
echo "</div>\n";

// 6. Implementation summary
echo "<h3>6. Implementation Summary</h3>\n";
echo "<div style='background: #e8f5e8; color: #2e7d32; padding: 15px; border-radius: 5px; margin: 10px 0;'>\n";
echo "<h4>✅ Changes Applied:</h4>\n";
echo "<ol>\n";
echo "<li><strong>jobs.php:</strong> Added deadline filtering to exclude expired jobs from listings</li>\n";
echo "<li><strong>index.php:</strong> Added deadline filtering to homepage job listings</li>\n";
echo "<li><strong>job-details.php:</strong> Added deadline validation in application process and UI</li>\n";
echo "</ol>\n";
echo "<p><strong>Result:</strong> Candidates can no longer see or apply to jobs that have passed their deadline.</p>\n";
echo "</div>\n";

?>

<style>
body { font-family: Arial, sans-serif; margin: 20px; line-height: 1.6; }
table { width: 100%; margin: 20px 0; border-collapse: collapse; }
th, td { padding: 8px; text-align: left; border: 1px solid #ddd; }
th { background-color: #f2f2f2; }
h1, h2, h3, h4 { color: #333; }
</style>