<?php
require_once 'config/config.php';

$database = new Database();
$db = $database->getConnection();

// Get search parameters
$search = isset($_GET['search']) ? sanitize($_GET['search']) : '';

// Build search query
$where_conditions = ["c.status = 'approved'"];
$params = [];

if ($search) {
    $where_conditions[] = "(c.name LIKE ? OR c.description LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

$where_clause = 'WHERE ' . implode(' AND ', $where_conditions);

// Get companies
$stmt = $db->prepare("
    SELECT c.*, COUNT(j.id) as job_count
    FROM companies c 
    LEFT JOIN jobs j ON c.id = j.company_id AND j.status = 'active'
    $where_clause
    GROUP BY c.id
    ORDER BY c.name ASC
");
$stmt->execute($params);
$companies = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Companies - <?php echo SITE_NAME; ?></title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/x-icon" href="assets/images/favicon.ico">
    <link rel="apple-touch-icon" sizes="180x180" href="assets/images/favicon.png">
    <link rel="icon" type="image/png" sizes="32x32" href="assets/images/favicon-32x32.png">
    <link rel="icon" type="image/png" sizes="16x16" href="assets/images/favicon-16x16.png">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <a class="navbar-brand" href="index.php">
                <i class="fas fa-briefcase me-2"></i><?php echo SITE_NAME; ?>
            </a>
            
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav me-auto">
                    <li class="nav-item">
                        <a class="nav-link" href="index.php">Home</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="jobs.php">Browse Jobs</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link active" href="companies.php">Companies</a>
                    </li>
                </ul>
                
                <ul class="navbar-nav">
                    <?php if (isLoggedIn()): ?>
                        <?php if ($_SESSION['user_type'] === 'admin'): ?>
                            <li class="nav-item">
                                <a class="nav-link" href="admin/dashboard.php">Admin Dashboard</a>
                            </li>
                        <?php elseif ($_SESSION['user_type'] === 'company'): ?>
                            <li class="nav-item">
                                <a class="nav-link" href="company/dashboard.php">Company Dashboard</a>
                            </li>
                        <?php elseif ($_SESSION['user_type'] === 'candidate'): ?>
                            <li class="nav-item">
                                <a class="nav-link" href="candidate/dashboard.php">My Profile</a>
                            </li>
                        <?php endif; ?>
                        <li class="nav-item">
                            <a class="nav-link" href="auth/logout.php">Logout</a>
                        </li>
                    <?php else: ?>
                        <li class="nav-item">
                            <a class="nav-link" href="auth/login.php">Login</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="auth/register.php">Register</a>
                        </li>
                    <?php endif; ?>
                </ul>
            </div>
        </div>
    </nav>

    <div class="container py-4">
        <!-- Page Header -->
        <div class="row mb-4">
            <div class="col-md-8">
                <h1 class="h2 mb-2">Our Companies</h1>
                <p class="text-muted">Discover amazing companies and their job opportunities</p>
            </div>
            <div class="col-md-4">
                <form method="GET">
                    <div class="input-group">
                        <input type="text" class="form-control" name="search" 
                               value="<?php echo htmlspecialchars($search); ?>" 
                               placeholder="Search companies...">
                        <button class="btn btn-primary" type="submit">
                            <i class="fas fa-search"></i>
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Companies Grid -->
        <div class="row">
            <?php if (empty($companies)): ?>
                <div class="col-12">
                    <div class="text-center py-5">
                        <i class="fas fa-building fa-4x text-muted mb-3"></i>
                        <h3>No companies found</h3>
                        <p class="text-muted">Try adjusting your search criteria.</p>
                        <a href="companies.php" class="btn btn-primary">View All Companies</a>
                    </div>
                </div>
            <?php else: ?>
                <?php foreach ($companies as $company): ?>
                    <div class="col-lg-4 col-md-6 mb-4">
                        <div class="card company-card h-100 hover-card">
                            <div class="card-body">
                                <div class="d-flex align-items-start mb-3">
                                    <div class="company-logo me-3">
                                        <?php if ($company['logo']): ?>
                                            <img src="<?php echo UPLOAD_URL . 'logos/' . $company['logo']; ?>" 
                                                 alt="Logo" class="company-logo">
                                        <?php else: ?>
                                            <div class="company-logo bg-light d-flex align-items-center justify-content-center">
                                                <i class="fas fa-building text-muted fa-2x"></i>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                    <div class="flex-grow-1">
                                        <h5 class="card-title mb-1"><?php echo htmlspecialchars($company['name']); ?></h5>
                                        <?php if ($company['website']): ?>
                                            <a href="<?php echo htmlspecialchars($company['website']); ?>" 
                                               target="_blank" class="text-muted text-decoration-none">
                                                <small><i class="fas fa-external-link-alt me-1"></i><?php echo htmlspecialchars($company['website']); ?></small>
                                            </a>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                
                                <?php if ($company['description']): ?>
                                    <p class="card-text"><?php echo substr(htmlspecialchars($company['description']), 0, 150) . (strlen($company['description']) > 150 ? '...' : ''); ?></p>
                                <?php endif; ?>
                                
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <small class="text-muted">
                                        <i class="fas fa-briefcase me-1"></i><?php echo $company['job_count']; ?> active job<?php echo $company['job_count'] != 1 ? 's' : ''; ?>
                                    </small>
                                    <small class="text-muted">
                                        <i class="fas fa-calendar me-1"></i>Joined <?php echo formatDate($company['created_at']); ?>
                                    </small>
                                </div>
                                
                                <?php if ($company['address']): ?>
                                    <div class="mb-3">
                                        <small class="text-muted">
                                            <i class="fas fa-map-marker-alt me-1"></i><?php echo htmlspecialchars($company['address']); ?>
                                        </small>
                                    </div>
                                <?php endif; ?>
                                
                                <div class="d-flex gap-2">
                                    <a href="company-details.php?id=<?php echo $company['id']; ?>" class="btn btn-outline-primary btn-sm flex-grow-1">
                                        <i class="fas fa-info-circle me-1"></i>View Details
                                    </a>
                                    <?php if ($company['job_count'] > 0): ?>
                                        <a href="jobs.php?company=<?php echo $company['id']; ?>" class="btn btn-primary btn-sm">
                                            <i class="fas fa-briefcase me-1"></i>View Jobs
                                        </a>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <!-- Call to Action -->
        <?php if (!isLoggedIn()): ?>
            <div class="row mt-5">
                <div class="col-12">
                    <div class="card bg-primary text-white">
                        <div class="card-body text-center py-5">
                            <h3 class="card-title">Are You a Company?</h3>
                            <p class="card-text mb-4">Companies are registered by our administrators. Please contact us to get your company added to our platform.</p>
                            <div class="d-flex justify-content-center gap-3">
                                <a href="auth/register.php" class="btn btn-light btn-lg">
                                    <i class="fas fa-user me-2"></i>Register as Job Seeker
                                </a>
                                <a href="mailto:admin@recruitment.com" class="btn btn-outline-light btn-lg">
                                    <i class="fas fa-envelope me-2"></i>Contact for Company Registration
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <!-- Footer -->
    <footer class="bg-dark text-white py-4 mt-5">
        <div class="container">
            <div class="row">
                <div class="col-lg-6">
                    <h5><?php echo SITE_NAME; ?></h5>
                    <p class="text-muted">Connecting talented individuals with amazing opportunities.</p>
                </div>
                <div class="col-lg-6">
                    <div class="text-lg-end">
                        <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
                    </div>
                </div>
            </div>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>